// abstracting classes from the XBeeNetwork to a general DataNetwork
// part of SenseWorld (prefix SW)
// developed for the SenseStage project

SWDataNetwork{
	var <spec;
	var <nodes;
	var <>verbose=0; // 1 is warning, 2 is all;

	var <>osc;

	var <>expectedNodes;
	//	var <>expectedSize;
	var <watcher;
	var <worrytime = 60; // time after which to worry whether node is still active
	var <>gui;
	var <>baseGui;

	var <>recTask;
	var <logfile;
	var <reclines = 0;
	var <>recnodes;

	var <recTime = false;
	var <timelogfile;

	*new{ 
		^super.new.init;
	}

	init{
		expectedNodes = Array.new;
		//		expectedSize = IdentityDictionary.new;
		nodes = IdentityDictionary.new;
		spec = SWDataNetworkSpec.new( this );
		watcher = SkipJack.new(
			{
				var now = Process.elapsedTime;
				nodes.do{ |it,i| 
					if ( it.elapsed > worrytime,
						{
							if ( verbose > 0, { "restarting network".postln; });
							it.restartAction.value;
						});
				};
			}, worrytime/10, name: "DataNetwork-watcher", autostart: false );
		recTask = Task.new( {} );
		this.watch( false );
	}

	/// --------- NODE control ---------

	checkDataType{ |data|
		var type;
		//		data.postcs;
		if ( data.first.isNil ){
			type = -1;
		};
		if ( data.first.isKindOf( SimpleNumber ) ){
			type = 0;
		};
		if ( data.first.isKindOf( Symbol ) or: data.first.isKindOf( String ) ){
			type = 1;
		};
		if ( type.isNil ){ type = 0; }; // proper error catching later
		^type;
	}

	setData{ |id,data|
		var ret = true;
		var ret2;
		var returnCode; // success;
		var lasttime;
		var node,type;
		if ( verbose > 1, { [id,data].postln; } );
		node = nodes[id];

		
		if ( node.isNil, {
			type = this.checkDataType( data );
			ret = this.registerNode( id, data.size, type );
			node = nodes[id];
			if ( verbose > 0 ) { ("registering node"+id+ret).postln; };
		});
		if ( ret ) {
			if ( recTime ){
				lasttime = node.lasttime;
			};
			ret2 = node.data_( data );
			if ( ret2 ){
				returnCode = 0;
				if ( recTime ){
					this.writeTimeUpdate( id, node.lasttime - lasttime );
				};
				if ( osc.notNil, {
					//	osc.sendData( id, data );
					osc.sendDataNode( node );
				});
			}{
				returnCode = 2; // wrong number of slots;
				if ( verbose > 1 ) { "wrong number of slots".postln; };
			}
		}{
			returnCode = 1; // error registering node;
		};
		^returnCode;
	}

	/* This should not be necessary:
	setDataSlot{ |id,id2,value|
		var ret = true;
		if ( verbose > 1, { [id,id2,value].postln; } );
		if ( nodes[id].isNil, {
			ret = false;
			if ( verbose > 0 ) { ("node"+id+"does not exist").postln; };
		});
		if ( ret ) { 
			nodes[id].dataSlot_( id2, value );
			if ( osc.notNil, {
				osc.sendData( id, nodes[id].data );
			});
		};
	}
	*/

	registerNode{ |id,sz,type|
		var ret,key,nnode;
		if ( type == -1){
			ret = false;
		}{
			ret = (sz > 0) and: (expectedNodes.indexOf( id ).notNil);
		};
		if ( ret ) {
			if ( type == 0 ){
				nnode = SWDataNode.new( id,sz );
			};
			if ( type == 1 ){
				nnode = SWDataStringNode.new( id,sz );
			};
			if ( nnode.notNil ){
				nodes.put( id, nnode );
			//	nodes.postcs;
			//	nodes[id].postcs;
				if ( osc.notNil, {
					osc.newNode( nnode );
				});
				if ( gui.notNil, {
					gui.addNode( nnode );
				});
				key = spec.findNode( id );
				if ( key.notNil, {this.at( key ).key = key; });
				sz.do{ |it| 
					key = spec.findSlot( id, it );
					if ( key.notNil, {this.at( key ).key = key; });
				};
			}{
				if ( verbose > 0 , {("node with id"+id+"has unknown type"+type ).postln;});
				ret = 13; // error code for wrong type
			};
		}{
			if ( verbose > 0 , {("node with id"+id+"and size"+sz+"is not expected to be part of the network" ).postln;});
		};
		^ret;
	}

	removeNode{ |id|
		nodes.removeAt( id );
		if ( osc.notNil, {
			osc.nodeRemoved( id );
		});
	}

	/// -------- expected nodes -----------

	addExpected{ |id,label,size=nil,type=0|
		if ( this.isExpected( id ).not, {
			expectedNodes = expectedNodes.add( id );
		});
		if ( label.notNil and: (label.asSymbol != \0 ), {
			this.add( label, id );
		},{
			// maybe the label is already in the spec
			label = spec.findNode( id );
		});
		if ( osc.notNil, {
			osc.newExpected( id, label );
		});
		if ( size.notNil, {
			if ( type == 0 ){
				this.setData( id, Array.fill( size, 0 ) );
			}{
				this.setData( id, Array.fill( size, {'0'} ) );
			};
		});
	}

	isExpected{ |id|
		^expectedNodes.indexOf( id ).notNil;
	}

	//---- spec (labeling and so on) ---------

	setSpec{ |name|
		spec.fromFile( name );
	}

	// direct access to spec:

	add{ |key, slot|
		var ns;
		spec.add( key, slot );
		if ( osc.notNil, {
			ns = this.at( key );
			//	ns.postln;
			if ( ns.isKindOf( SWDataNode ),{
				osc.newNode( ns );
			});
			if ( ns.isKindOf( SWDataSlot ),{
				osc.newSlot( ns );
			});
		});
	}

	// ---- named access to nodes, values and actions -------

	at{ |key|
		^spec.at( key );
	}

	value{ |key|
		^spec.value( key );
	}

	/* THIS should not be allowed! use setData instead.
	value_{ |key,value|
		spec.value_( key, value );
	}
	*/

	action_{ |key,action|
		spec.action_( key, action );
	}

	//---- -- node bus control -------------

	bus{ |key|
		^spec.bus( key );
	}

	createBus{ |key,server|
		spec.createBus( key, server );
	}

	freeBus{ |key|
		spec.freeBus( key );
	}

	createAllBuses{ |server|
		spec.createAllBuses( server );
	}

	freeAllBuses{
		spec.freeAllBuses;
	}

	createAllNodeBuses{ |server|
		nodes.do{ |it| it.createBus(server) };
	}

	// --------- data input control -------

	worrytime_{ |wt|
		worrytime = wt;
		watcher.dt = wt/10;
	}

	watch{ |onoff=true|
		if ( onoff, { watcher.start }, { watcher.stop; } );
	}


	// ------- data logging and recording --------

	// recording
	initRecord{ |fn,dt=0.025,stamp=false,break=36000|
		//		var recordnodes;
		fn = fn ? "SWDataNetworkLog";

		fn = fn++"_"++Date.localtime.stamp;

		// save a spec if none is given:
		if ( spec.name.isNil ){
			spec.save( fn );
		}{
			spec.save;
		};

		logfile = MultiFileWriter.new( fn ++ ".txt" );
		logfile.stringMethod = \asCompileString;
		//	logfile =  File(fn++"_"++Date.localtime.stamp++".txt", "w");
				//		recnodes = this.writeHeader;

		// copies the spec to the directory into which we are writing our log
		spec.copyFile( logfile.pathDir );

		// save a tab readable version of the spec:
		spec.saveAsTabs( logfile.pathDir +/+ logfile.fileName ++ "_header.txt");

		recTask = Task.new( {
			loop {
				logfile.open;
				logfile.curFile.timeStamp = stamp;
				this.writeHeader;
				//	recnodes.dump;
				break.do{ |it|
					this.writeLine( dt );
					dt.wait;
				};
				logfile.close;
			}
		} );
	}

	record{ |onoff|
		if ( onoff ) {
			recTask.reset.play;
		}{
			recTask.stop;
			logfile.close;
		};
	}

	closeRecord{
		this.record( false );
		//	logfile.close;
		recTask = Task.new( {} );
	}

	writeHeader{
		var recslots;
		// this tells the spec used for the recording:
		logfile.writeLine( [spec.name] );
		//		logfile.write( "\n" );
		//		logfile.write( "time\t" );

		recnodes = nodes.collect{ |node|
			//			node.slots.do{ |it| logfile.write( it.id.asCompileString ); logfile.write( "\t" ); };
			node.id;
		}.asArray;

		// this creates a header with the ids of the node slots
		recslots = recnodes.collect{ |nodeid|
			nodes[nodeid].slots.collect{ |it| it.id };
		}.flatten;
		
		//	logfile.write( "\n" );

		logfile.writeLine( [ "time" ] ++ recslots );
		//	^recordnodes;
	}

	writeLine{ |dt|
		var data;
		/*
			logfile.write( dt.asString );
			logfile.write( "\t" );
			recordnodes.do{ |it|
			nodes[it].slots.collect{ |slot| slot.value }.do{ |dat|
			logfile.write( dat.asCompileString );
			logfile.write( "\t" );
			}; 
			};
			logfile.write( "\n" );
		*/
		data = recnodes.collect{ |it|
			nodes[it].slots.collect{ |slot| slot.logvalue }
		}.flatten; 

		logfile.writeLine( [dt] ++ data);
	}


	/// log update times

	// recording
	initTimeRecord{ |fn,stamp=false|
		fn = fn ? "SWDataNetworkUpdateLog";
		timelogfile =  TabFileWriter(fn++"_"++Date.localtime.stamp++".txt").timeStamp_(stamp);
		//		timelogfile =  File(fn++"_"++Date.localtime.stamp++".txt", "w");
		recTime = true;
	}

	writeTimeUpdate{ |id,time|
		timelogfile.writeLine( [id,time]);
		/*
		timelogfile.write( id.asString );
		timelogfile.write( "\t" );
		timelogfile.write( time.asString );
		timelogfile.write( "\n" );
		*/
	}

	closeTimeRecord{
		recTime = false;
		timelogfile.close;
	}


	// ------------ Debugging -------------

	debug_{ |onoff|
		nodes.do{ |sl|
			sl.do{ |slt| slt.debug_( onoff ) } };
	}

	// add interfaces:

	makeGui{
		^SWDataNetworkGui.new( this );
	}

	makeLogGui{
		^SWDataNetworkLogGui.new( this );
	}

	makeBasicGui{
		^SWDataNetworkBaseGui.new( this );
	}

	addOSCInterface{
		^SWDataNetworkOSC.new( this );
	}

}


// a DataNode are a collection of slots which are physically connected to each other, e.g. data gathered from the same device.
SWDataNode{

	var <id;
	var <>key;
	var <type = 0;

	var <slots;
	var <data;
	var <>scale = 1;
	var <lasttime;

	var <>action;
	var <>restartAction;

	var >bus;
	var <>databus;

	//	var <>trigger;

	// monitoring support
	var <busmonitor;

	*new{ |id,maxs=4|
		^super.new.init(id,maxs);
	}
	
	init{ |ident,maxs|
		id = ident;
		lasttime = 0;
		slots = Array.fill( maxs, 0 );
		data = Array.fill( maxs, 0.0 );
		// the restart action should contain what should be done if the node does not provide data anymore
		restartAction = {};
		action = {};
		lasttime = Process.elapsedTime;
		//		trigger = {};
		this.initSlots;
	}

	initSlots{
		slots.do{ |it,i| slots.put( i, SWDataSlot.new([id,i]) ); };
	}

	// -------- slots and data -------

	data_{ |indata|
		if ( indata.size == slots.size , {
			data = indata.asFloat * scale;
			data.do{ |it,i| slots[i].value = it };
			action.value( data, this );
			this.setLastTime;
			//	trigger.value;
			^true;
		});
		^false;
		//		indata.copyRange(0,data.size-1).do{ |it,i| data[i].value = it };
	}

	/*
	dataSlot_{ |id,indata|
		if( id < slots.size ){
			data[id] = indata * scale;
			slots[id].value = data[id];
			lasttime = Process.elapsedTime;
			action.value( data );
		};
	}
	*/


	value{
		^data;
	}

	setLastTime{
		lasttime = Process.elapsedTime;
	}

	elapsed{
		^(Process.elapsedTime - lasttime );
	}

	// --- Bus support ---

	createBus{ |s|
		if ( this.bus.isNil, {
			s = s ? Server.default;
			databus = DataBus.new( { slots.collect{ |it| it.value } }, slots.size, s )
		});
	}

	bus{
		if ( bus.notNil, { ^bus } );
		if ( databus.isNil, { ^nil } );
		^databus.bus;
	}

	freeBus{
		if ( bus.notNil, { bus.free; bus = nil; },
			{
			if ( databus.notNil, { databus.free; databus = nil; });
			});
	}

// JITLib support
	kr{
		var b;
		this.createBus;
		b = this.bus;
		^In.kr( b, b.numChannels );
	}

	// ---------- debugging and monitoring -------

	debug_{ |onoff|
		slots.do{ |sl|
			sl.do{ |slt| slt.debug_( onoff ) } };
	}

	monitor{ |onoff=true|
		if ( onoff, {
			if ( busmonitor.isNil, { 
				if ( bus.isNil, { this.createBus } );
				busmonitor = BusMonitor.new( this.bus );
			});
			busmonitor.start;
		}, { busmonitor.stop; });
	}

	monitorClose{
		busmonitor.cleanUp;
		busmonitor = nil;
	}

}

SWDataSlot{
	var <>id;
	var <>key;
	var <type = 0;

	var <value;
	var <>action;

	var <bus;
	var debugAction;

	var <>scale=1;
	var <map;
	var <range;

	// monitoring support
	var <busmonitor;

	*new{ |id|
		^super.new.init(id);
	}

	init{ |ident|
		id = ident;
		action = {};
		debugAction = {};
		value = 0;
	}

	// ------ data and action -----
	value_{ |val|
		value = val * scale;
		// map to control spec from input range after scaling
		if ( map.notNil, { value = map.map( range.unmap( value ) ) } );
		action.value( value );
		debugAction.value( value );
		if ( bus.notNil, { bus.set( value ) } );
	}

	logvalue{
		^value;
	}


	/// -------- scaling, mapping and calibrating --------

	map_{ |mp|
		if( range.isNil){
			// input range after scaling:
			range = [0,1].asSpec;
		};
		map = mp;
	}

	range_{ |mp|
		if( map.isNil){
			// input range after scaling:
			map = [0,1].asSpec;
		};
		range = mp;
	}

	// currently only does minimum:
	calibrate{ |steps=100| // about two seconds currently
		var calib,values;
		values = Array.new( steps );
		range = [0,1].asSpec;
		calib = Routine{ 
			var mean;
			steps.do{ |it,i| values.add( this.value ); it.yield; };
			mean = values.sum / values.size;
			range.minval = mean;
			this.debug_( false );
			"calibration done".postln;
		};
		debugAction = { calib.next };
	}

	// --- bus support ----

	createBus{ |s|
		s = s ? Server.default;
		if ( bus.isNil, {
			bus = Bus.control( s, 1 );
		},{
			if ( bus.index.isNil, {
				bus = Bus.control( s, 1 );
			});
		});
	}

	freeBus{
		bus.free;
		bus = nil;
	}

// JITLib support
	kr{
		this.createBus;
		^In.kr( bus );
	}

	/// ------- debugging and monitoring ------

	debug_{ |onoff|
		if ( onoff, {
			debugAction = { |val| [ id, value, key ].postln; };
		},{
			debugAction = {};
		});
	}


	monitor{ |onoff=true|
		if ( onoff, {
			if ( busmonitor.isNil, { 
				this.createBus;
				busmonitor = BusMonitor.new( this.bus );
			});
			busmonitor.start;
		}, { busmonitor.stop; });
	}

	monitorClose{
		busmonitor.cleanUp;
		busmonitor = nil;
	}

}



SWDataNetworkSpec{
	classvar <>all,<folder;

	var <>name;
	var <map, network;

	*initClass { 
		// not yet used
		this.makeSaveFolder;
		this.loadSavedInfo;
		all	= all ? Set.new;
	}

	*loadSavedInfo{
		all = (folder+/+"allspecs.info").load;
	}
	
	*makeSaveFolder { 
		var testfile, testname = "zzz_datanetwork_test_delete_me.txt"; 
		folder = (Platform.userAppSupportDir +/+ "DataNetworkSpecs").standardizePath;
		testfile = File(folder +/+ testname, "w");

		if (testfile.isOpen.not) 
			{ unixCmd("mkdir" + folder.escapeChar($ )) }
			{ testfile.close;  unixCmd("rm" + folder.escapeChar($ ) +/+ testname) }
	}

	*saveAll{
		var file, res = false;
		var filename;
		filename = folder +/+ "allspecs.info";
		file = File(filename, "w"); 
		if (file.isOpen) { 
			res = file.write(all.asCompileString);
			file.close;
		};
		^res;
	}


	*new { |netw|
		^super.new.init(netw);
	}

	init{ |netw|
		network = netw;
		map = IdentityDictionary.new;
	}

	saveAsTabs{ |path|
		var file,mynodes,myslots;
		file = TabFileWriter.new( path, "w" );
		
		mynodes = network.nodes.collect{ |node|
			node.id;
		}.asArray;

		// this creates a header with the ids of the node slots
		myslots = mynodes.collect{ |nodeid|
			network.nodes[nodeid].slots.collect{ |it| [it.id,it.key] };
		}.flatten.flop;
		file.writeLine( myslots[0] );
		file.writeLine( myslots[1] );
		file.close;
	}

	save{ |name|
		var file, res = false;
		var filename;
		all.add( name.asSymbol );
		this.name = name ? this.name;
		filename = folder +/+ name ++ ".spec";
		file = File(filename, "w"); 
		if (file.isOpen) { 
			res = file.write(map.asCompileString);
			file.close;
		};
		this.class.saveAll;
		^res;
	}

	fromFile { |name| 
		var slot;
		this.name = name;
		map = (folder +/+ name++".spec").load;
		map.keysValuesDo{ |key,it|
			slot = this.at( key );
			if ( slot.notNil, { slot.key = key; } );
		}
	}

	copyFile{ |target|
		("cp"+(folder +/+ name++".spec")+target).unixCmd;
	}

	fromFileName { |fn| 
		var slot;
		this.name = name;
		map = (fn++".spec").load;
		map.keysValuesDo{ |key,it|
			slot = this.at( key );
			if ( slot.notNil, { slot.key = key; } );
		}
	} 

	findNode{ |id|
		^map.findKeyForValue( id );
	}

	findSlot{ |id1,id2|
		var keySlot = nil;
		map.keysValuesDo{ |key,val| if ( val == [id1,id2] ) { keySlot = key } };
		^keySlot;
		//		^map.findKeyForValue( [id1,id2] );
	}



	// --- the methods below can all be accessed from the network directly ----

	add{ |key, slot|
		map.put( key, slot );
		if ( this.at( key ).notNil, {
			this.at( key ).key = key;
		});
	}

	// ------- named access to nodes, slots

	// returns the slot or node
	at{ |key|
		var id1,id2;
		var item;
		item = map.at( key );
		if ( item.isKindOf( Array ), {
			id1 = map.at(key)[0];
			id2 = map.at(key)[1];
			if ( network.nodes[id1].isNil, { ^nil } );
			^network.nodes[id1].slots[id2];
		},{
			^network.nodes[item]
		});
		// map.at(key)
	}

	value{ |key|
		^this.at(key).value;
	}

	/* // this should not be allowed!
	value_{ |key,value|
		var slot;
		slot = this.at(key);
		slot.value_(value);
		^slot;
	}
	*/

	action_{ |key,action|
		var slot;
		slot = this.at(key);
		slot.action_(action);
		^slot;		
	}

	//-------- node bus control ---------

	bus{ |key|
		^this.at(key).bus;
	}

	createBus{ |key,server|
		this.at( key ).createBus( server );
	}

	freeBus{ |key|
		this.at( key ).freeBus;
	}


	createAllBuses{ |server|
		map.do{ |it|
			if ( it.isKindOf( Array ), {
				network.nodes.at( it[0] ).slots.at( it[1] ).createBus( server );
			},{
				network.nodes.at( it ).createBus( server );
			});
		};
	}

	freeAllBuses{
		map.do{ |it|
			network.nodes.at( it[0] ).slots.at( it[1] ).freeBus;
		};
	}


	/*	setAllActions{ |action|
		map.do{ |it|
			device.slots.at( it[0] ).at( it[1] ).action_( action );
		};
		}*/


}
